<?php

namespace App\Http\Controllers;
use App\Events\MessageSentEvent;
use App\Models\Gig;
use App\Models\Order;
use Illuminate\Http\Request;
use App\Events\OrderPlacedNotification;
use App\Models\Notification;
use App\Models\OrderSubmission;
use App\Models\Message;


class OrderController extends Controller
{

    // public function store(Request $request)
    // {
    //     $request->validate([
    //         'gig_id' => 'required|exists:gigs,id',
    //         'buyer_name' => 'required|string|max:255',
    //         'email' => 'required|email',
    //         'project_details' => 'required|string',
    //         'budget' => 'nullable|numeric|min:0',
    //         'expected_delivery_date' => 'required|date|after_or_equal:today',
    //     ]);

    //     $order = Order::create([
    //         'buyer_id' => $request->user()->id,
    //         'gig_id' => $request->gig_id,
    //         'buyer_name' => $request->buyer_name,
    //         'email' => $request->email,
    //         'project_details' => $request->project_details,
    //         'budget' => $request->budget,
    //         'expected_delivery_date' => $request->expected_delivery_date,
    //     ]);

    //     return response()->json([
    //         'message' => 'Order placed successfully.',
    //         'order' => $order,
    //     ]);
    // }



    public function store(Request $request)
    {
        $request->validate([
            'gig_id' => 'required|exists:gigs,id',
            'buyer_name' => 'required|string|max:255',
            'email' => 'required|email',
            'project_details' => 'required|string',
            'budget' => 'nullable|numeric|min:0',
            'expected_delivery_date' => 'required|date|after_or_equal:today',
        ]);
    
        // Create the order
        $order = Order::create([
            'buyer_id' => $request->user()->id,
            'gig_id' => $request->gig_id,
            'buyer_name' => $request->buyer_name,
            'email' => $request->email,
            'project_details' => $request->project_details,
            'budget' => $request->budget,
            'expected_delivery_date' => $request->expected_delivery_date,
        ]);
    
        // Find the gig and get the creator (owner) ID
        $gig = Gig::findOrFail($request->gig_id);
        $creator_id = $gig->user_id;
    
        // Create the notification for the gig creator, including project details
        $notification = Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $creator_id,
            'message' => 'You received a new order from ' . $request->buyer_name . '. Project details: ' . $request->project_details,
            'order_id' => $order->id,
        ]);
    
        // Log the result of the notification creation
        if ($notification) {
            \Log::info('Notification successfully created for order ID: ' . $order->id);
        } else {
            \Log::error('Failed to create notification for order ID: ' . $order->id);
        }
    
        return response()->json([
            'message' => 'Order placed successfully.',
            'order' => $order,
        ]);
    }
    
    public function showOrder($orderId)
    {
        // Fetch the order by ID
        $order = Order::findOrFail($orderId);

        // Return the view with the order data
        return view('order', compact('order'));
    }


    // public function respondToOrder(Request $request, $orderId)
    // {
    //     $request->validate([
    //         'action' => 'required|in:accept,reject',
    //     ]);

    //     $order = Order::findOrFail($orderId);

    //     // Only gig owner (creator) can respond
    //     $gig = Gig::findOrFail($order->gig_id);
    //     if ($gig->user_id !== $request->user()->id) {
    //         return response()->json(['message' => 'Unauthorized.'], 403);
    //     }

    //     // Update status
    //     $status = $request->action === 'accept' ? 'accepted' : 'rejected';
    //     $order->status = $status;
    //     $order->save();

    //     // Notify buyer
    //     $creator = $request->user();
    //     $message = "Your order for gig \"{$gig->title}\" has been {$status} by {$creator->name}.";

    //     Notification::create([
    //         'sender_id' => $creator->id,
    //         'receiver_id' => $order->buyer_id,
    //         'message' => $message,
    //     ]);

    //     return response()->json([
    //         'message' => "Order has been {$status}.",
    //         'order' => $order,
    //     ]);
    // }



    public function respondToOrder(Request $request, $orderId)
    {
        $request->validate([
            'action' => 'required|in:accept,reject',
        ]);

        $order = Order::findOrFail($orderId);

        // Only gig owner (creator) can respond
        $gig = Gig::findOrFail($order->gig_id);
        if ($gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        // Update status
        $status = $request->action === 'accept' ? 'accepted' : 'rejected';
        $order->status = $status;
        $order->save();

        // Notify buyer
        $creator = $request->user();
        $message = "Your order for gig \"{$gig->title}\" has been {$status} by {$creator->name}.";

        Notification::create([
            'sender_id' => $creator->id,
            'receiver_id' => $order->buyer_id,
            'message' => $message,
        ]);

        return response()->json([
            'message' => "Order has been {$status}.",
            'order' => $order,
        ]);
    }



    public function myOrders(Request $request)
    {
        $userId = $request->user()->id;

        $orders = Order::whereHas('gig', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })
        ->where('status', 'accepted')
        ->with(['gig', 'buyer'])
        ->get();

        return response()->json([
            'orders' => $orders,
        ]);
    }

    public function updateOrderDetails(Request $request, $orderId)
    {
        $request->validate([
            'label' => 'nullable|string|max:255',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'milestone' => 'nullable|in:initial,draft,revisions,final',
        ]);
    
        $order = Order::findOrFail($orderId);
        $gig = $order->gig;
    
        if ($gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
    
        $order->update([
            'label' => $request->label,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'milestone' => $request->milestone,
        ]);
    
        Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $order->buyer_id,
            'message' => "Order details have been updated.",
        ]);
    
        return response()->json([
            'message' => 'Order updated and buyer notified successfully.',
            'order' => $order,
        ]);
    }
    

    public function updateMilestone(Request $request, $id)
    {
        $request->validate([
            'milestone' => 'required|in:initial,draft,revisions,final',
        ]);
    
        $order = Order::findOrFail($id);
        $gig = $order->gig;
    
        if ($gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
    
        $order->milestone = $request->milestone;
        $order->save();
    
        Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $order->buyer_id,
            'message' => "Milestone updated to '{$order->milestone}'.",
        ]);
    
        return response()->json([
            'message' => 'Milestone updated and buyer notified successfully.',
            'order' => $order,
        ]);
    }
    

    public function updateProgressStatus(Request $request, $id)
    {
        $request->validate([
            'progress_status' => 'required|in:pending,inprogress,completed',
        ]);
    
        $order = Order::findOrFail($id);
        $gig = $order->gig;
    
        if ($gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
    
        $order->progress_status = $request->progress_status;
        $order->save();
    
        // Notify Buyer
        Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $order->buyer_id,
            'message' => "Order progress updated to '{$order->progress_status}'.",
        ]);
    
        return response()->json([
            'message' => 'Progress status updated and buyer notified successfully.',
            'order' => $order,
        ]);
    }

    // public function sendMessageToBuyer(Request $request, $id)
    // {
    //     $request->validate([
    //         'message' => 'required|string',
    //     ]);

    //     $order = Order::findOrFail($id);

    //     if ($order->gig->user_id !== $request->user()->id) {
    //         return response()->json(['message' => 'Unauthorized'], 403);
    //     }

    //     $message = Message::create([
    //         'sender_id' => $request->user()->id,
    //         'receiver_id' => $order->buyer_id,
    //         'order_id' => $order->id,
    //         'message' => $request->message,
    //     ]);

    //     return response()->json([
    //         'message' => 'Message sent successfully',
    //         'data' => $message,
    //     ]);
    // }

    public function sendMessageToBuyer(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string',
        ]);

        $order = Order::findOrFail($id);

        if ($order->gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Dispatch the event instead of directly creating the message
        event(new MessageSentEvent(
            $order->id, 
            $request->user()->id, 
            $order->buyer_id, 
            $request->message
        ));

        // $message = Message::create([
        //     'sender_id' => $request->user()->id,
        //     'receiver_id' => $order->buyer_id,
        //     'order_id' => $order->id,
        //     'message' => $request->message,
        // ]);
        // event(new MessageSentEvent($message)); 

        return response()->json([
            'message' => 'Message sent successfully',
        ]);
    }


    // public function sendMessageToSeller(Request $request, $id)
    // {
    //     $request->validate([
    //         'message' => 'required|string',
    //     ]);
    
    //     $order = Order::findOrFail($id);
    
    //     if ($order->buyer_id !== $request->user()->id) {
    //         return response()->json(['message' => 'Unauthorized'], 403);
    //     }
    
    //     $message = Message::create([
    //         'sender_id' => $request->user()->id,
    //         'receiver_id' => $order->gig->user_id,
    //         'order_id' => $order->id,
    //         'message' => $request->message,
    //     ]);
    
    //     return response()->json([
    //         'message' => 'Reply sent successfully',
    //         'data' => $message,
    //     ]);
    // }


    public function sendMessageToSeller(Request $request, $id)
{
    $request->validate([
        'message' => 'required|string',
    ]);

    $order = Order::findOrFail($id);

    if ($order->buyer_id !== $request->user()->id) {
        return response()->json(['message' => 'Unauthorized'], 403);
    }

    // Dispatch the event instead of directly creating the message
    event(new MessageSentEvent(
        $order->id, 
        $request->user()->id, 
        $order->gig->user_id, 
        $request->message
    ));



    // $message = Message::create([
    //     'sender_id' => $request->user()->id,
    //     'receiver_id' => $order->buyer_id,
    //     'order_id' => $order->id,
    //     'message' => $request->message,
    // ]);

    // // Dispatch event if you're using Laravel Echo or Pusher
    // broadcast(new MessageSentEvent($message))->toOthers();

    return response()->json([
        'message' => 'Reply sent successfully',
    ]);
}

    



    // Upload Work for Order
    public function uploadWork(Request $request, $id)
    {
        $request->validate([
            'file' => 'required|file|mimes:zip,pdf,docx,jpg,png',
            'message' => 'nullable|string',
        ]);
    
        $order = Order::findOrFail($id);
    
        if ($order->gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
    
        $originalName = time() . '_' . $request->file('file')->getClientOriginalName();
        $filePath = $request->file('file')->storeAs("uploads/order_{$order->id}", $originalName);
    
        // Log or save submission
        $submission = OrderSubmission::create([
            'order_id' => $order->id,
            'file_path' => $filePath,
            'message' => $request->message,
        ]);
    
        // Notify buyer
        Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $order->buyer_id,
            'message' => 'New work has been uploaded for your order.',
        ]);
    
        return response()->json([
            'message' => 'Work uploaded and buyer notified.',
            'submission' => $submission,
        ]);
    }
    

    public function viewSubmission(Request $request, $id)
    {
        $order = Order::findOrFail($id);

        if ($order->gig->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $submission = OrderSubmission::where('order_id', $order->id)->latest()->get();

        return response()->json([
            'message' => 'Work fetched successfully',
            'data' => $submission,
        ]);
    }


    public function showOrderForBuyer(Request $request, $orderId)
    {
        $order = Order::with(['gig', 'buyer'])->findOrFail($orderId);

        // Check if the order belongs to the current user (the buyer)
        if ($order->buyer_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return response()->json([
            'message' => 'Order details fetched successfully.',
            'order' => $order,
        ]);
    }

    public function getOrderProgressForBuyer(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the current user (the buyer)
        if ($order->buyer_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return response()->json([
            'message' => 'Order progress status fetched successfully.',
            'order' => $order->progress_status,
        ]);
    }


        // Fetch the order details for the buyer
        public function getOrderDetails(Request $request, $orderId)
        {
            // Fetch the order by ID
            $order = Order::where('id', $orderId)
                ->where('buyer_id', $request->user()->id)  // Ensure that the buyer is the one requesting the order
                ->with('gig', 'buyer')  // Eager load the related gig and buyer
                ->first();
    
            // If order not found, return an error response
            if (!$order) {
                return response()->json(['message' => 'Order not found or you do not have permission to view this order.'], 404);
            }
    
            // Return the order details
            return response()->json([
                'message' => 'Order details fetched successfully.',
                'order' => $order,
            ]);
        }
        

        public function getMyOrders(Request $request)
        {
            $user = auth()->user();
        
            if (!$user->hasRole('Buyer')) {
                return response()->json([
                    'status' => false,
                    'message' => 'Unauthorized access.'
                ], 403);
            }
        
            $orders = Order::with(['gig', 'contentCreator'])
                ->where('buyer_id', $user->id)
                ->get();
        
            return response()->json([
                'status' => true,
                'message' => 'Your orders retrieved successfully.',
                'data' => $orders
            ]);
        }
        

        public function getOrdersFromBuyer(Request $request, $buyerId)
{
    $creatorId = $request->user()->id;

    // Fetch orders where the gig belongs to the creator and the buyer matches
    $orders = Order::where('buyer_id', $buyerId)
        ->whereHas('gig', function ($query) use ($creatorId) {
            $query->where('user_id', $creatorId);
        })
        ->with(['gig', 'buyer']) // optional: to include gig and buyer details
        ->get();

    return response()->json([
        'orders' => $orders,
    ]);
}

        

}
